
/*

 	SVMGlue.java. Author: Richard Johansson (richard@cs.lth.se)
   
    SimpleSVM: A wrapper for LIBSVM-style software.

    Copyright (C) 2004 - 2005, Richard Johansson (richard@cs.lth.se).

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
    Created in 2005 by Richard Johansson (richard@cs.lth.se).
   
 	$Log: SVMGlue.java,v $
 	Revision 1.5  2005/10/27 11:50:23  richard
 	Added the LGPL preamble.
 	
 	Revision 1.4  2005/08/29 08:22:38  richard
 	Added methods for probability predictions.
 	

 */


package simplesvm;

import java.io.*;

class SVMGlue {

	static {
		System.loadLibrary("svmglue");
	}

	private static String tempDir = "/tmp/"; //null;
	
	private static native long load_model(String filename);
	
	private static native int svm_predict(long model_ptr,
			int[] positions, double[] values);
	
	private static native void svm_destroy_model(long model_ptr);
	
	private static native int[] svm_get_labels(long model_ptr);

	private static native long svm_create_prob_array(long model_ptr);
	
	private static native void svm_predict_probabilities(long model_ptr, int[] positions,
														 double[] values, long prob_array_ptr,
														 double[] probs);
	
	private long prob_array_ptr;
	private long model_ptr;
	
	public SVMGlue(String filename) {
		model_ptr = load_model(filename);
		if(model_ptr == 0)
			throw new RuntimeException("Could not load model: " +filename);
	}

	public SVMGlue(InputStream is) {
		if(tempDir == null)
			throw new RuntimeException("Temporary directory must be set.");
		int rand = new java.util.Random().nextInt();
		String filename = tempDir + "svmglue_" + rand;
		try {
			OutputStream out = new FileOutputStream(filename);
			while(is.available() > 0) {
			        int av = is.available();
				byte[] buf = new byte[av];
				int rb_tot = 0;
				do {
				    int rb = is.read(buf, 0, av - rb_tot);
				    out.write(buf, 0, rb);
				    rb_tot += rb;
				} while(rb_tot < av); 
			}
			out.close();
		} catch(Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Got exception!");
		}
		model_ptr = load_model(filename);
		try {
			if(!new File(filename).delete())
				System.err.println("Warning: could not remove temporary file " + filename);
		} catch(Exception e) {
			System.err.println("Warning: could not remove temporary file " + filename);
		}
		if(model_ptr == 0)
			throw new RuntimeException("Could not load model: " +filename);
	}
	
	protected void finalize() {
		svm_destroy_model(model_ptr);
	}
	
	public int predict(int[] positions, double[] values) {
		return svm_predict(model_ptr, positions, values);
	}

	public void initProbs() {
		prob_array_ptr = svm_create_prob_array(model_ptr);
	}
	
	public void predictProbabilities(int[] positions, double[] values, double[] probs) {
		svm_predict_probabilities(model_ptr, positions, values, prob_array_ptr, probs);
	}

	public int[] getLabels() {
		return svm_get_labels(model_ptr);
	}
 	
	public static void main(String[] argv) {
		SVMGlue glue = new SVMGlue("heart_scale.model");
		
		//System.out.println(glue.predict(new int[] { 1, 2 }, new double[] { 0.3, 0.6 }));

		int[] xxx = SVMGlue.svm_get_labels(glue.model_ptr);
		for(int i = 0; i < xxx.length; i++)
			System.out.println(xxx[i]);

		glue.initProbs();
		
		double[] yyy = new double[2];

		for(int i = 0; i < 100000; i++)
			glue.predictProbabilities(new int[] { 1, 2 }, new double[] { 0.3, 0.6 }, yyy);

		for(int i = 0; i < yyy.length; i++)
			System.out.println(yyy[i]);
		
	}
	
	public static void setTempDir(String dir) {
		if(!dir.endsWith("/"))
			dir += "/";
		tempDir = dir;
	}
}
