/* 
    SimpleSVM.java.

    SimpleSVM: A wrapper for LIBSVM-style software.

    Copyright (C) 2004 - 2005, Richard Johansson (richard@cs.lth.se).

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
    Created in 2004 by Richard Johansson (richard@cs.lth.se).
 
 	$Log: SimpleSVM.java,v $
 	Revision 1.12  2005/10/27 15:10:30  richard
 	Fixed a bug of number encoding.
 	
 	Revision 1.11  2005/10/27 12:35:06  richard
 	Changed the input format (removed the first and last pipe).
 	
 	Revision 1.10  2005/10/27 11:50:33  richard
 	Added the LGPL preamble.
 	
 	Revision 1.9  2005/10/27 11:17:51  richard
 	Probabilities when loading non-natively.
 	
 	Revision 1.8  2005/09/21 12:56:17  richard
 	Moved some printout to stderr.
 	
 	Revision 1.7  2005/09/09 13:44:56  richard
 	Added probability output to the svms. Added the dumptarget option.
 	
 	Revision 1.6  2005/06/15 08:44:03  richard
 	Pruning and printing encodings.
 	
 	Revision 1.5  2005/06/14 07:00:11  richard
 	Added the setTempDir and some printouts.
 	
 	Revision 1.4  2005/06/13 14:20:45  richard
 	Added a jar file constructor.
 	
 	Revision 1.3  2005/06/13 14:12:57  richard
 	Changed the constructors.
 	
 	Revision 1.2  2005/06/13 12:39:24  richard
 	Changed the constructors.
 	 
 */

package simplesvm;

import java.util.*;
import java.io.*;
import java.util.zip.GZIPInputStream;
import libsvm.*;

import java.util.jar.*;

/* TODO: null */

/**
 * @author Richard Johansson (richard@cs.lth.se)
 *
 */
public class SimpleSVM {
	
	static class Boolean {
		private boolean b;
		public Boolean(boolean b) {
			this.b = b;
		}
		public boolean booleanValue() {
			return b;
		}
		public static boolean parseBoolean(String s) {
			s = s.trim();
			if(s.equals("true"))
				return true;
			if(s.equals("false"))
				return false;
			throw new RuntimeException("Could not parse boolean");
		}
		
	}
	
	static final String STR_SEP = "\\#";
	
	public static class SVMTypes {
		public static final int C_SVC = 0;
		public static final int NU_SVC = 1;
		public static final int ONE_CLASS_SVM = 2;
		public static final int EPSILON_SVR = 3;
		public static final int NU_SVR = 4;
	}
	
	public static class SVMKernelTypes {
		public static final int LINEAR = 0;
		public static final int POLYNOMIAL = 1;
		public static final int RADIAL_BASIS = 2;
		public static final int SIGMOID = 3;
	}
	
	private static final String SVM_TRAIN_EXTERNAL_COMMAND = "./svm-train";
	
	private static final int SVM_TYPE = SVMTypes.C_SVC;
	private static final int SVM_KERNEL_TYPE = SVMKernelTypes.RADIAL_BASIS;	
	private static final double C = 512;
	private static final double gamma = 0.001;
	
	/* 
	 Encoding of values: 
	 
	 BOOLEANS are 1 if true, 0 if false.
	 NUMBERS are scaled so that the training instances are in [0..1].
	 STRINGS are coded with boolean flags for each occurring symbol.
	 
	 OUTPUT VALUES are given a number.
	 */
	
	public static final String NULL = "<E>";
	
	private static Map newMap() {
		return new HashMap();
		//return new ImplicitHashMap(1);
	}
	
	public static class SVMClassifier {

		private svm_model model;
		private SVMGlue glue;
		
		private SVMEncoding encoding;

		private int[] labels;
		private double[] probs;
		private HashMap<Integer, Integer> labelToIndex;
		
		private void initModel(String modelName, boolean useNative) {
			try {
				if(useNative) {
					System.err.println("Loading model (native)...");
					glue = new SVMGlue(modelName);
					glue.initProbs();
					labels = glue.getLabels();
					initLabels();
					probs = new double[labels.length];
					System.err.println("Done.");
				} else {
					System.err.println("Loading model (java)...");
					model = svm.svm_load_model(modelName);
					System.err.println("Done.");
				}
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Got exception.");
			}
		}
		
		private void initModel(InputStream modelIS, boolean useNative) {
			try {
				if(useNative) {
					System.err.println("Loading model (native)...");
					glue = new SVMGlue(modelIS);
					glue.initProbs();
					labels = glue.getLabels();
					initLabels();
					probs = new double[labels.length];
					System.err.println("Done.");
				} else {
					System.err.println("Loading model (java)...");
					model = svm.svm_load_model(modelIS);
					System.err.println("Done.");
				}
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Got exception.");
			}
		}

		private void initLabels() {
			labelToIndex = new HashMap<Integer, Integer>();
			for(int i = 0; i < labels.length; i++)
				labelToIndex.put(labels[i], i);
		}
		
		public SVMClassifier(InputStream modelIS, SVMEncoding enc,
				boolean useNative) {
			initModel(modelIS, useNative);
			encoding = enc;
		}

		public SVMClassifier(String modelName, SVMEncoding enc,
				boolean useNative) {
			initModel(modelName, useNative);
			encoding = enc;

		}

		public SVMClassifier(InputStream modelIS, InputStream encIS, 
				boolean useNative) {
			initModel(modelIS, useNative);
			encoding = new SVMEncoding(encIS);
		}
		
		public SVMClassifier(String modelName, InputStream encIS, 
				boolean useNative) {
			initModel(modelName, useNative);
			encoding = new SVMEncoding(encIS);
		}

		public SVMClassifier(String modelName, String encodingName,
				boolean useNative) {
			initModel(modelName, useNative);
			try {
				encoding = new SVMEncoding(encodingName);
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Got exception.");
			}
		}
		
		public SVMClassifier(String modelName, String encodingName) {
			this(modelName, encodingName, false);
		}

		public SVMClassifier(JarFile jarfile, String modelName, 
				String encodingName, boolean useNative)	{
			try {
				JarEntry modelEntry = jarfile.getJarEntry(modelName);
				InputStream mIS = jarfile.getInputStream(modelEntry);
				JarEntry encEntry = jarfile.getJarEntry(encodingName);
				InputStream eIS = jarfile.getInputStream(encEntry);
				initModel(mIS, useNative);
				encoding = new SVMEncoding(eIS);
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Got exception.");
			}
			
		}
		
		private svm_node[] encode(Object[] pattern) {
			Pair p = encoding.encodeStrings(pattern);
			
			int[] positions = (int[]) p.left;
			double[] values = (double[]) p.right;
			
			svm_node[] x = new svm_node[positions.length];
			for(int j = 0; j < positions.length; j++) {
				x[j] = new svm_node();
				x[j].index = positions[j];
				x[j].value = values[j];
			}
			
			return x;
		}
		
		// 	private double classify(int[] positions, double[] values) {
		// 	    svm_node[] x = new svm_node[positions.length];
		// 	    for(int j = 0; j < positions.length; j++) {
		// 		x[j] = new svm_node();
		// 		x[j].index = positions[j];
		// 		x[j].value = values[j];
		// 	    }
		// 	    return svm.svm_predict(model, x);
		// 	}
		
		public Object classifyStrings(Object[] pattern) {
			int answer;
			if(glue != null) {
				Pair p = encoding.encodeStrings(pattern);
				
				int[] positions = (int[]) p.left;
				double[] values = (double[]) p.right;
				
				answer = glue.predict(positions, values);
			} else {
				svm_node[] x = encode(pattern);
				answer = (int) svm.svm_predict(model, x);
			}
			return encoding.outvalues[answer];
		}

		private void computeProbsAux(Object[] pattern) {
			if(glue != null) {
				Pair p = encoding.encodeStrings(pattern);
				
				int[] positions = (int[]) p.left;
				double[] values = (double[]) p.right;
				
				glue.predictProbabilities(positions, values, probs);
			} else {
				svm_node[] x = encode(pattern);
				svm.svm_predict_probability(model, x, probs);
			}
		}

		public double[] computeProbsRestricted(Object[] pattern, List allowedLabels) {
			//System.out.println("pattern = " + Arrays.asList(pattern));
			//System.out.println("allowedLabels = " + allowedLabels);
			
			double[] out = new double[allowedLabels.size()];
			computeProbsAux(pattern);
			int count = 0;
			double sum = 0;

			//System.out.println("encoding.outvaluenumbers = " + encoding.outValueNumbers);

			for(Iterator i = allowedLabels.iterator(); i.hasNext(); ) {
				Object label = i.next();

				//System.out.println("label = " + label);

				// Frst fr vi svmkodnings-siffran fr label
				Integer i1 = (Integer) encoding.outValueNumbers.get(label);
				//System.out.println("i1 = " + i1);
				
				if(i1 != null) {
					Integer i2 = labelToIndex.get(i1);
					
					// Sedan fr vi positionen i slh-vektorn fr denna siffra
					//System.out.println("i2 = " + i2);
					if(i2 != null) {
						//int index2 = labelToIndex.get(index1);					
						out[count] = probs[i2];
					} else // Aldrig observerad vid trning, dvs finns inte i labels
						out[count] = 0;
				} else // Finns inte i kodningen
					out[count] = 0;
				
				//if(out[count] < MIN_PROB)
					//out[count] = MIN_PROB;
				
				sum += out[count];
				count++;
			}
			for(int i = 0; i < out.length; i++)
				out[i] /= sum;
			return out;
		}
		
		public SVMEncoding getEncoding() {
			return encoding;
		}
		
		//private double[] d = new double[1];
		
		/*
		public double activation(Object[] pattern) {
			svm_node[] x = encode(pattern);
			svm.svm_predict_values(model, x, d);
			return d[0];
		}*/
		
	}
	
	public static class SVMEncoding {
		public int n_features;
		public Map[] maps;
		public double[] mins, maxs, scalings;
		public Map outValueNumbers;
		public Object[] outvalues;
		public Class[] classes;
		public int[] indices;
		
		public SVMEncoding(int n_features) {
			this.n_features = n_features;
			maps = new Map[n_features];
			for(int j = 0; j < n_features; j++)
				maps[j] = newMap();
			
			mins = new double[n_features];
			for(int j = 0; j < n_features; j++)
				mins[j] = Double.MAX_VALUE;
			
			maxs = new double[n_features];
			for(int j = 0; j < n_features; j++)
				maxs[j] = Double.NEGATIVE_INFINITY;
			
			scalings = new double[n_features];
			
			classes = new Class[n_features];
			
			outValueNumbers = new HashMap();
		}
		
		public SVMEncoding(InputStream s) {
			init(s);
		}
		
		private void init(InputStream s) {
			try {
				System.err.println("Loading encoding...");
				ObjectInputStream is = new ObjectInputStream(s);
				n_features = is.readInt();
				maps = (Map[]) is.readObject();
				mins = (double[]) is.readObject();
				scalings = (double[]) is.readObject();
				outValueNumbers = (Map) is.readObject();
				outvalues = (Object[]) is.readObject();
				classes = (Class[]) is.readObject();
				indices = (int[]) is.readObject();
				
				is.close();
				System.err.println("Done.");
			} catch(Exception e) {
				e.printStackTrace();
				System.exit(1);
			}
		}
		
		public SVMEncoding(String s) throws IOException {
			if(s.endsWith("gz"))
				init(new GZIPInputStream(new FileInputStream(s)));
			else
				init(new FileInputStream(s));
		}		
		
		void setIndices() {
			indices = new int[n_features];
			indices[0] = 1;
			for(int i = 1; i < n_features; i++) {
				Class c = classes[i-1];
				if(c == Boolean.class || c == Float.class 
						|| c == Double.class || c == Integer.class)
					indices[i] = indices[i-1] + 1;
				else
					indices[i] = indices[i-1] + maps[i-1].size();
			}
		}
		
		void setScalings() {
			for(int i = 0; i < n_features; i++)
				scalings[i] = 1 / (maxs[i] - mins[i]);
		}
		
		void findOutValues() {
			outvalues = new Object[outValueNumbers.size()];
			for(Iterator i = outValueNumbers.entrySet().iterator(); i.hasNext(); ) {
				Map.Entry e = (Map.Entry) i.next();
				int index = ((Integer) e.getValue()).intValue();
				outvalues[index] = e.getKey();
			}
		}
		
		private void writeToFile(String filename) {
			System.out.println("Writing encoding to file...");
			try {
				ObjectOutputStream out 
				= new ObjectOutputStream(new FileOutputStream(filename));
				
				out.writeInt(n_features);
				out.reset();
				out.writeObject(maps);
				out.reset();
				out.writeObject(mins);
				out.reset();
				out.writeObject(scalings);
				out.reset();
				out.writeObject(outValueNumbers);
				out.reset();
				out.writeObject(outvalues);
				out.reset();
				out.writeObject(classes);
				out.reset();
				out.writeObject(indices);
				out.reset();
				
				out.close();
				
			} catch(Exception e) {
				e.printStackTrace();
				System.exit(1);
			}
			System.out.println("Finished writing encoding.");
		}
		
		public Pair encodeStrings(Object[] objs) {
			if(objs.length != n_features) {
				throw new RuntimeException("Illegal number of features. "
						+ "Expected " + n_features + ", found " 
						+ (objs.length));
				
			}
			ArrayList l = new ArrayList();			
			for(int j = 0; j < n_features; j++)
				if(classes[j] == String.class) {
					String s = (String) objs[j];
					if(s.equals(NULL)) {
						
					} else {
						String[] tokens = s.split(STR_SEP);
						for(int k = 0; k < tokens.length;k++){
							Integer index = (Integer) maps[j].get(tokens[k]);
							if(index != null) {
								Pair p = new Pair(new Integer(index.intValue() + indices[j]), new Double(1));
								l.add(p);
							}
						}
					}
				} else if(classes[j] == Boolean.class) {
					boolean b = Boolean.parseBoolean((String) objs[j]);
					if(b) {
						Pair p = new Pair(new Integer(indices[j]),
								new Double(1));
						l.add(p);
					}
				} else if(classes[j] == Double.class) {
					double d = Double.parseDouble((String) objs[j]);
					d -= mins[j];
					d *= scalings[j];
					Pair p = new Pair(new Integer(indices[j]),
							new Double(d));
					l.add(p);
				} else if(classes[j] == Integer.class) {
					double d = (double) Integer.parseInt((String) objs[j]);
					d -= mins[j];
					d *= scalings[j];
					Pair p = new Pair(new Integer(indices[j]),
							new Double(d));
					l.add(p);
				} else if(classes[j] == Float.class) {
					double d = (double) Float.parseFloat((String) objs[j]);
					d -= mins[j];
					d *= scalings[j];
					Pair p = new Pair(new Integer(indices[j]),
							new Double(d));
					l.add(p);
				}
				
			int[] positions = new int[l.size()];
			double[] values = new double[l.size()];
			for(int i = 0; i < l.size(); i++) {
				Pair p = (Pair) l.get(i);
				positions[i] = ((Integer) p.left).intValue();
				values[i] = ((Double) p.right).doubleValue();
			}
			return new Pair(positions, values);
		}
		
		public int encodeTarget(Object t) {
			Integer i = (Integer) outValueNumbers.get(t);
			return i.intValue();
		}

		void dump() {
			System.out.println("SVM Encoding:");
			System.out.println("Nbr of features: " + n_features);
			System.out.println("Nbr of target values: " + outvalues.length);
			int nkeys = 0;
			for(int i = 0; i < n_features; i++) {
				String clname = classes[i].getName().replaceAll("[a-zA-Z0-9]+(\\.|\\$)", "");
				System.out.print("Feature " + i + ": " + clname);
				System.out.print(", index = " + indices[i]);
				if(maps[i] != null) {
					System.out.print(", " + maps[i].size() + " keys");
					nkeys += maps[i].size();
				}
				System.out.println();
			}
			System.out.println("Total number of keys: " + nkeys);
		}
		
		void dumptargets() {
			int nkeys = 0;
			for(int i = 0; i < outvalues.length; i++) {
				System.out.println("" + i + ": " + outvalues[i]);
			}
		}
		
		void prune(String filename, String outfilename) {
			try {
				System.out.println("Pruning:");

				System.out.println("Reversing maps...");
				HashMap rev = new HashMap();
				for(int i = 0; i < n_features; i++) {
					if(maps[i] != null) {
						for(Iterator it = maps[i].entrySet().iterator(); it.hasNext();) {
							Map.Entry en = (Map.Entry) it.next();
							int index = ((Integer) en.getValue()).intValue() + indices[i];
							rev.put(new Integer(index), 
									new Pair(new Integer(i), en.getKey()));
						}
					}
				}

				for(int i = 0; i < n_features; i++) {
					if(maps[i] != null)
						maps[i] = newMap();
				}
				
				System.out.println("Traversing file...");
				BufferedReader in = new BufferedReader(new FileReader(filename));
				String line = in.readLine();
				while(line != null) {
					String[] tokens = line.split(" ");
					for(int i = 1; i < tokens.length; i++) {
						String[] ss = tokens[i].split(":");
						int index = Integer.parseInt(ss[0]);
						Pair p = (Pair) rev.get(new Integer(index));
						if(p != null) {
							int f = ((Integer) p.left).intValue();
							maps[f].put(p.right, new Integer(index - indices[f]));
						}
					}
					line = in.readLine();
				}
				System.out.println("Writing new encoding...");
				writeToFile(outfilename);
				System.out.println("Done.");
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Got exception.");
			}
		}
		
	}
	
	public static void processExamplesFromFile(String exampleInFile, 
			String exampleOutFile,
			String encodingOutFile) {
		processExamplesFromFile(exampleInFile, exampleOutFile,
				encodingOutFile, null);
	}
	
	public static void processExamplesFromFile(String exampleInFile, 
			String exampleOutFile,
			String encodingOutFile,
			Map predefined) {
		
		System.out.println("Reading examples from a file...");
		
		//ArrayList l = new ArrayList();
		
		SVMEncoding enc = null;
		
		boolean[] saw_nonnumber = null, saw_nonboolean = null, 
		saw_strsep = null;
		
		try {
			BufferedReader reader = new BufferedReader(new FileReader(exampleInFile));
			String line = reader.readLine();
			int nlines = 0;
			while(line != null) {
				nlines++;
				if(nlines % 1000 == 0)
					System.out.println("nlines = " + nlines);

				//System.out.println(line);
				
				line = line.trim();
				//line = line.substring(1, line.length() - 1);
				String[] tokens = line.split("\\|");
				
				if(enc == null) {
					int n_features = tokens.length - 1;
					enc = new SVMEncoding(n_features);
					
					if(predefined != null)
						for(Iterator i = predefined.entrySet().iterator(); 
						i.hasNext(); ) {
							Map.Entry e = (Map.Entry) i.next();
							enc.outValueNumbers.put(e.getKey(), e.getValue());
						}
					
					saw_nonnumber = new boolean[n_features];
					saw_nonboolean = new boolean[n_features];
					saw_strsep = new boolean[n_features];
					
				} else
					if(tokens.length - 1 != enc.n_features)
						throw new RuntimeException("Illegal number of features. "
								+ "Expected " + enc.n_features + ", found " 
								+ (tokens.length - 1) + " in line nbr " + nlines + ": " + line);
				for(int j = 0; j < enc.n_features; j++) {
					
					String[] ts2 = tokens[j+1].split(STR_SEP);
					
					if(ts2.length != 1)
						saw_strsep[j] = true;
					
					for(int k = 0; k < ts2.length; k++) {
						if(!ts2[k].matches("true|false"))
							saw_nonboolean[j] = true;
						if(!ts2[k].matches("(\\-)?[0-9]+(\\.[0-9]+)?"))
							saw_nonnumber[j] = true;
						addToIndexMap(enc.maps[j], ts2[k]);

					}

					if(!saw_nonnumber[j]) {
						double val = Double.parseDouble(tokens[j+1]);
						if(val > enc.maxs[j])
							enc.maxs[j] = val;
						if(val < enc.mins[j])
							enc.mins[j] = val;
					}
						
				}
				addToIndexMap(enc.outValueNumbers, tokens[0]);
				
				line = reader.readLine();
			}
			
		} catch(Exception e) {
			e.printStackTrace();
			System.exit(1);
		}
		
		//System.out.println(Arrays.asList(enc.maps));
		
		/* Guess types. */
		for(int i = 0; i < enc.n_features; i++) {
			if(!saw_nonboolean[i] && !saw_strsep[i]) {
				enc.classes[i] = Boolean.class;
				enc.maps[i] = null;
			} else if(!saw_nonnumber[i] && !saw_strsep[i]) {
				enc.classes[i] = Double.class;
				enc.maps[i] = null;
			} else
				enc.classes[i] = String.class;
			//System.out.println(enc.classes[i]);
		}
		
		enc.setScalings();
		enc.setIndices();
		enc.findOutValues();
		
		processFile(enc, exampleInFile, exampleOutFile);
		
		enc.writeToFile(encodingOutFile);
		
		System.out.println("Finished processing the examples.");
	}
	
	public static void processFile(SVMEncoding enc, String exampleInFile,
			String exampleOutFile) {
		System.out.println("Encoding the examples...");
		try {
			BufferedReader reader = new BufferedReader(new FileReader(exampleInFile));
			PrintWriter out = new PrintWriter(new FileWriter(exampleOutFile));
			
			String[] fvect = new String[enc.n_features];
			
			String line = reader.readLine();
			while(line != null) {
				line = line.trim();
				//line = line.substring(1, line.length() - 1);
				
				String[] tokens = line.split("\\|");
				int outvalue = enc.encodeTarget(tokens[0]);
				System.arraycopy(tokens, 1, fvect, 0, fvect.length);
				Pair p = enc.encodeStrings(fvect);
				
				int[] positions = (int[]) p.left;
				double[] values = (double[]) p.right;
				
				out.print(outvalue);
				
				for(int i = 0; i < positions.length; i++)
					out.print(" " + positions[i] + ":" + values[i]);
				out.println();
				
				line = reader.readLine();
			}
			
			out.close();
			
		} catch(Exception e) {
			e.printStackTrace();
			System.exit(1);
		}	
		System.out.println("Finished encoding the examples.");
	}
	
	private static void addToIndexMap(Map m, Object key) {
		Object index = m.get(key);
		if(index == null) {
			index = new Integer(m.size());
			m.put(key, index);
		}
	}
	
	private static void executeTrainerExternal(String filename) {
		String cmd = SVM_TRAIN_EXTERNAL_COMMAND + " -s " + SVM_TYPE + " -t " + SVM_KERNEL_TYPE 
		+ " -c " + C + " -g " + gamma + " " + filename;
		
		System.out.println("Executing external trainer...");
		System.out.println("Command line: " + cmd);
		
		try {
			Process process = Runtime.getRuntime().exec(cmd);
			process.waitFor();
			int exitValue = process.exitValue();
			System.out.println("exitValue = " + exitValue);
		} catch(Exception e) {
			e.printStackTrace();
			System.exit(1);
		}
		
		System.out.println("Trainer finished.");
	}
	
//	private static void executeTrainerJava(String filename) {
//		String[] argv = new String[] { "-s", "" + SVM_TYPE, "-t", "" + SVM_KERNEL_TYPE,
//				"-c", "" + C, "-g", "" + gamma, filename
//		};
//		System.out.println("Executing Java trainer...");
//		System.out.println("Arguments:" + Arrays.asList(argv));
//		try {
//			new svm_train().run(argv);
//		} catch(Exception e) {
//			e.printStackTrace();
//		}
//		System.out.println("Trainer finished.");
//	}
	
	public static void main(String[] argv) {
		main_new(argv);
	}
	
	public static void main_new(String[] argv) {

		if(argv.length >= 2 && argv[0].equals("-dumpencoding")) {
			try {
				SVMEncoding enc = new SVMEncoding(argv[1]);
				enc.dump();
			} catch(Exception e) {
				e.printStackTrace();
				System.exit(1);
			}

			System.exit(0);
		}
		if(argv.length >= 2 && argv[0].equals("-dumptargets")) {
			try {
				SVMEncoding enc = new SVMEncoding(argv[1]);
				enc.dumptargets();
			} catch(Exception e) {
				e.printStackTrace();
				System.exit(1);
			}

			System.exit(0);
		}
		
		if(argv.length >= 4 && argv[0].equals("-pruneencoding")) {
			try {
				SVMEncoding enc = new SVMEncoding(argv[1]);
				enc.prune(argv[2], argv[3]);
			} catch(Exception e) {
				e.printStackTrace();
				System.exit(1);
			}
			System.exit(0);
		}
		
		HashMap predefined = new HashMap();

		for(int i = 3; i < argv.length; i += 2) {
			String key = argv[i];
			try {
				Integer value = new Integer(Integer.parseInt(argv[i+1]));
				predefined.put(key, value);
			} catch(Exception e) {
				System.err.println("Could not read a number.");
				System.exit(1);
			}
		}
		
		processExamplesFromFile(argv[0], argv[1], argv[2], predefined);
	}
	
	static class Pair implements java.io.Serializable {
		public Object left, right;
		
		/**
		 * Constructor.
		 */
		public Pair(Object left, Object right) {
			this.left = left;
			this.right = right;
		}
		public String toString() {
			return "(" + left + ", " + right + ")";
		}
		public boolean equals(Object o) {
			if(!(o instanceof Pair))
				return false;
			Pair p = (Pair) o;
			return p.left.equals(left) && p.right.equals(right);
		}
		public int hashCode() {
			return 31 * left.hashCode() + right.hashCode();
		}
	}
	
	public static void setTempDir(String dir) {
		SVMGlue.setTempDir(dir);
	}
	
}
