/************* COPYRIGHT AND CONFIDENTIALITY INFORMATION ********************
**                                                                         **
** Copyright (c) 2010 technicolor                                          **
** All Rights Reserved                                                     **
**                                                                         **
** This program contains proprietary information which is a trade          **
** secret of technicolor and/or its affiliates and also is protected as    **
** an unpublished work under applicable Copyright laws. Recipient is       **
** to retain this program in confidence and is not permitted to use or     **
** make copies thereof other than as permitted in a written agreement      **
** with technicolor, UNLESS OTHERWISE EXPRESSLY ALLOWED BY APPLICABLE LAWS.**
**                                                                         **
****************************************************************************/

/** \file
 * Multimedia switch connection API.
 *
 * A multimedia switch connection is a source/sink of a multimedia stream.
 *
 * \version v1.0
 *
 *************************************************************************/
#ifndef  MMCONN_INC
#define  MMCONN_INC

/*########################################################################
#                                                                       #
#  HEADER (INCLUDE) SECTION                                             #
#                                                                       #
########################################################################*/
#include "mmconntypes.h"
#include "mmcommon.h"

/*########################################################################
#                                                                       #
#  MACROS/DEFINES                                                       #
#                                                                       #
########################################################################*/

#ifndef RTPPACKET_GET_SEQNUM
#define RTPPACKET_GET_SEQNUM(rtp)    ((rtp)->seq[0] << 8 | (rtp)->seq[1]);  /**< RtpHeader helper macro: Get sequence number field within RTP header. */
#endif

#ifndef RTPPACKET_SET_SEQNUM
#define RTPPACKET_SET_SEQNUM(rtp, val)                       \
    ((rtp)->seq[0] = (unsigned char)(((val) & 0xff00) >> 8), \
     (rtp)->seq[1] = (unsigned char)((val) & 0x00ff));                      /**< RtpHeader helper macro: Set sequence number field within RTP header. */
#endif

#ifndef RTPPACKET_GET_TIMESTAMP
#define RTPPACKET_GET_TIMESTAMP(rtp)           \
    ((rtp)->ts[0] << 24 | (rtp)->ts[1] << 16 | \
     (rtp)->ts[2] << 8 | (rtp)->ts[3]);                                     /**< RtpHeader helper macro: Get timestamp field within RTP header. */
#endif

#ifndef RTPPACKET_SET_TIMESTAMP
#define RTPPACKET_SET_TIMESTAMP(rtp, val)                        \
    ((rtp)->ts[0] = (unsigned char)(((val) & 0xff000000) >> 24), \
     (rtp)->ts[1] = (unsigned char)(((val) & 0x00ff0000) >> 16), \
     (rtp)->ts[2] = (unsigned char)(((val) & 0x0000ff00) >> 8),  \
     (rtp)->ts[3] = (unsigned char)((val) & 0x000000ff));                   /**< RtpHeader helper macro: Set timestamp field within RTP header. */
#endif

#ifndef RTPPACKET_GET_SSRC
#define RTPPACKET_GET_SSRC(rtp)                    \
    ((rtp)->ssrc[0] << 24 | (rtp)->ssrc[1] << 16 | \
     (rtp)->ssrc[2] << 8 | (rtp)->ssrc[3]);                                 /**< RtpHeader helper macro: Get SSRC field within RTP header. */
#endif

#ifndef RTPPACKET_SET_SSRC
#define RTPPACKET_SET_SSRC(rtp, val)                               \
    ((rtp)->ssrc[0] = (unsigned char)(((val) & 0xff000000) >> 24), \
     (rtp)->ssrc[1] = (unsigned char)(((val) & 0x00ff0000) >> 16), \
     (rtp)->ssrc[2] = (unsigned char)(((val) & 0x0000ff00) >> 8),  \
     (rtp)->ssrc[3] = (unsigned char)((val) & 0x000000ff));                 /**< RtpHeader helper macro: Set SSRC field within RTP header. */
#endif


/*########################################################################
#                                                                       #
#  TYPES                                                                #
#                                                                       #
########################################################################*/

/**
 * RTP header template, can be used to interprete data.
 */
typedef struct {
#ifdef __BIG_ENDIAN 
    unsigned char  version:2;   /**< protocol version */
    unsigned char  p:1;         /**< padding flag */
    unsigned char  x:1;         /**< header extension flag */
    unsigned char  cc:4;        /**< CSRC count */
    unsigned char  m:1;         /**< marker bit */
    unsigned char  pt:7;        /**< payload type */
#else
    unsigned char  cc:4;        /**< CSRC count */
    unsigned char  x:1;         /**< header extension flag */
    unsigned char  p:1;         /**< padding flag */
    unsigned char  version:2;   /**< protocol version */
    unsigned char  pt:7;        /**< payload type */
    unsigned char  m:1;         /**< marker bit */  
#endif
    unsigned char  seq[2];      /**< 16-bit sequence number */
    unsigned char  ts[4];       /**< 32-bit timestamp */
    unsigned char  ssrc[4];     /**< synchronization source */
} MmConnRtpHeader;

/**
 * Multimedia connection packet types
 */

typedef enum {
    MMCONN_PACKET_TYPE_RAW   =  0,  /**< RAW media. */
    MMCONN_PACKET_TYPE_RTP   ,  /**< RTP media. */
    MMCONN_PACKET_TYPE_RTCP  ,  /**< RTCP media. */
    MMCONN_PACKET_TYPE_UDPTL ,  /**< UDPTL media. */
} MmConnPacketType;

/**
 * Multimedia connection packet header description.
 */
typedef struct {
    MmConnPacketType type; /**< Packet type */
} MmConnPacketHeader;

/**
 * Multimedia event types.
 */
typedef enum {
    MMCONN_EV_INGRESS_RTP_MEDIA = 0,      /**< Incoming (connection -> switch) media. Parm will hold payload type */
    MMCONN_EV_INGRESS_MEDIA_TIMEOUT,      /**< No media received for a certain time (for IP connections). Parm has no meaning for this event. */
    MMCONN_EV_DONE,                       /**< Something is done. Parm has no meaning for this event. */
} MmConnEventType;

/**
 * Multimedia event types.
 */
typedef struct {
    MmConnEventType type;       /**< Event Type */
    int             parm;       /**< Optional parameter of event */
} MmConnEvent;


/**
 * Callback function to write data to an endpoint.
 *
 * An endpoint which wants to receive data from a connection instance of multimedia switch can implement this callback function which allows the connection to push data to the endpoint.
 *
 * \since v1.0
 *
 * \pre \c NONE
 *
 * \post The endpoint has handled the data push.
 *
 * \param [in] conn Handle of connection instance.
 * \param[in] data which is pushed to the endpoint.
 * \param [in] header Header used to identify packet.
 * \param [in] bytes Number of bytes which were pushed.
 *
 * \return ::MmPbxError
 * \retval MMPBX_ERROR_NOERROR The write action to the endpoint suceeded.
 * \todo Add other possible return values after implementation.
 */
typedef MmPbxError (*MmConnWriteCb)(MmConnHndl         conn,
                                    void               *data,
                                    MmConnPacketHeader *header,
                                    unsigned int       bytes);


/*########################################################################
#                                                                       #
#  FUNCTION PROTOTYPES                                                  #
#                                                                       #
########################################################################*/


/**
 * Write packet to a multimedia switch  connection instance.
 *
 * This function needs to be used to write data to a connection instance.
 *
 * \since v1.0
 *
 * \pre \c conn must be a valid handle.
 *
 * \post The content of \c buff has been written to the connection.
 *
 * \param [in] conn Handle of connection instance.
 * \param [in] header Header used to identify packet.
 * \param [in] buff Buffer which contains data to write.
 * \param [in] bytes Number of bytes to write.
 *
 * \return ::MmPbxError.
 * \retval MMPBX_ERROR_NOERROR Data successfully written.
 * \todo Add other possible return values after implementation.
 */
MmPbxError mmConnWrite(MmConnHndl         conn,
                       MmConnPacketHeader *header,
                       unsigned char      *buff,
                       unsigned int       bytes);

/**
 * Send an event to the user space counterpart of mmConn.
 *
 * This function can be used to send an event to the user space counterpart of mmConn.
 *
 * \since v1.0
 *
 * \pre \c mmConn must be a valid handle.
 *
 * \post The event has been end to the user space counterpart of mmConn.
 *
 * \param [in] mmConn Handle of connection instance.
 * \param [in] event Event to send.
 *
 * \return ::MmPbxError.
 * \retval MMPBX_ERROR_NOERROR Event successfully sent.
 * \todo Add other possible return values after implementation.
 */
MmPbxError mmConnSendEvent(MmConnHndl      mmConn,
                              MmConnEvent *event);

/**
 * Register write callback for a multimedia switch  connection instance.
 *
 * This callback function can be registered by an endpoint to get an event when data is pushed to the endpoint.
 *
 * \since v1.0
 *
 * \pre \c conn must be a valid handle.
 * \pre \c cb must be a valid callback function of type ::MmConnWriteCb.
 *
 * \post The endpoint has handled the event.
 *
 * \param [in] conn Handle of multimedia switch connection instance.
 * \param [in] cb Write callback function.
 *
 * \return ::MmPbxError.
 * \retval MMKSSWITCH_ERROR_NOERROR The callback function has been successfully registered.
 * \todo Add other possible return values after implementation.
 */
MmPbxError mmConnRegisterWriteCb(MmConnHndl    conn,
                                 MmConnWriteCb cb);

/**
 * Unregister write callback for a multimedia switch connection instance.
 *
 * This function unregisters a write callback function from a connection instance.
 *
 * \since v1.0
 *
 * \pre \c conn must be a valid handle.
 *
 * \post The callback function will no longer be called.
 *
 * \param [in] conn Handle of multimedia switch connection instance.
 * \param [in] cb Write callback function.
 *
 * \return ::MmPbxError.
 * \retval MMKSSWITCH_ERROR_NOERROR The callback function has been successfully unregistered.
 * \todo Add other possible return values after implementation.
 */
MmPbxError mmConnUnregisterWriteCb(MmConnHndl    conn,
                                   MmConnWriteCb cb);

#endif   /* ----- #ifndef MMCONN_INC  ----- */
