#include <errno.h>
#include <string.h>

#include "unittest/unittest.h"

#include "crc.h"

/* this the function under test */
int rip_proc_read(char  *page,
                  char  **start,
                  off_t offset,
                  int   count,
                  int   *eof,
                  void  *data);

/* rip2lib function mockups */
#include "rip2.h"

static unsigned long    drv_read_id_requested;
static int              drv_read_return;
static void             *drv_read_return_data;
static unsigned long    drv_read_return_length;

int rip2_drv_read(unsigned long *length,
                  T_RIP2_ID     id,
                  void          *data)
{
	drv_read_id_requested = id;
	if (drv_read_return_data && (drv_read_return_length > 0)) {
		memcpy(data, drv_read_return_data, drv_read_return_length);
		*length = drv_read_return_length;
	}
	return drv_read_return;
}

static void rip2_drv_read_return(int            rv,
                                 void           *data,
                                 unsigned long  length)
{
	drv_read_return         = rv;
	drv_read_return_data    = data;
	drv_read_return_length  = length;
}

static void setup(void)
{
	drv_read_return         = 0;
	drv_read_return_data    = NULL;
	drv_read_return_length  = 0;

	/* Initialise CRC32 */
	rip2_mk_crc32_table(CRC32, rip2_crc32_hw);
}

static void teardown(void)
{
}

START_TEST(item_not_found)
{
	int         rv;
	char        page[4 * 1024];
	char        *start  = NULL;
	int         eof     = -1;
	T_RIP2_ID   id      = 1;
	int         i;

	int error_values[] =
	{
		0,
		RIP2_ERR_INV_RIP,
		RIP2_ERR_NOMEM,
		RIP2_ERR_NOELEM,
		RIP2_ERR_BADCRC,
		RIP2_ERR_PERM,
		RIP2_ERR_BADCRYPTO,
		RIP2_SUCCESS + 1,
		9999,
	};

	for (i = 0; i < (sizeof(error_values) / sizeof(int)); i++, id++) {
		rip2_drv_read_return(error_values[i], NULL, 0);
		rv = rip_proc_read(page, &start, 0, 1024, &eof, &id);
		ck_assert_int_eq(rv, -EFAULT);
		ck_assert_int_eq(id, drv_read_id_requested);
	}
}
END_TEST

START_TEST(read_data)
{
	int         rv;
	char        page[4 * 1024];
	char        *start  = NULL;
	int         eof     = -1;
	T_RIP2_ID   id      = 1;
	int         offset;

	char    *TestData   = "TestReturnData";
	int     testlen     = strlen(TestData);

	/* read all data from offset to EOF */
	for (offset = 0; offset < testlen; offset++) {
		rip2_drv_read_return(RIP2_SUCCESS, TestData, testlen);
		rv = rip_proc_read(page, &start, offset, 1024, &eof, &id);
		ck_assert(rv > 0);
		ck_assert_int_eq(id, drv_read_id_requested);
		ck_assert_int_eq(eof, 1);
		ck_assert(start != NULL);
		ck_assert(strncmp(start, TestData + offset, 1) == 0);
		ck_assert_int_eq(start[rv - 1], '\n');
	}


	/* read a single byte at offset */
	for (offset = 0; offset < testlen; offset++) {
		rip2_drv_read_return(RIP2_SUCCESS, TestData, testlen);
		rv = rip_proc_read(page, &start, offset, 1, &eof, &id);
		ck_assert(rv > 0);
		ck_assert_int_eq(id, drv_read_id_requested);
		ck_assert_int_eq(eof, 0);
		ck_assert(start != NULL);
		ck_assert_int_eq(start[0], TestData[offset]);
	}

	/* read final newline */
	rip2_drv_read_return(RIP2_SUCCESS, TestData, testlen);
	rv = rip_proc_read(page, &start, testlen, 1, &eof, &id);
	ck_assert(rv > 0);
	ck_assert_int_eq(id, drv_read_id_requested);
	ck_assert_int_eq(eof, 1);
	ck_assert(start != NULL);
	ck_assert_int_eq(start[0], '\n');

	/* read beyond EOF */
	rip2_drv_read_return(RIP2_SUCCESS, TestData, testlen);
	rv = rip_proc_read(page, &start, testlen + 1, 1, &eof, &id);
	ck_assert_int_eq(rv, 0);
	ck_assert_int_eq(id, drv_read_id_requested);
	ck_assert_int_eq(eof, 1);
}

END_TEST

static singleTestCaseInfo testcases[] =
{
	{ .name = "item not found", .function = item_not_found },
	{ .name = "read_data",      .function = read_data      },
	{ }
};

void register_procrip_tests(Suite *s)
{
	TCase *tc = tcase_create("/proc/rip unittest");

	tcase_addtests(tc, testcases);
	// Must come after tcase_addtests() because that one also adds
	// a fixture that enables memleak tracing. If not, the code in
	// setup() is not traced and you can miss memleaks.
	tcase_add_checked_fixture(tc, setup, teardown);
	suite_add_tcase(s, tc);
}
